package com.nsftools.charthelper;

/*
 * copyright 2009 SNAPPS, Julian Robichaux, and Rob McDonagh
 * 
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 2.1 of the License, or
 *     (at your option) any later version.
 *                 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *     
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with this program.  If not, see http://www.gnu.org/licenses.
 */


import java.io.*;
import java.util.Iterator;

import org.jfree.chart.*;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.data.general.Dataset;
import org.jfree.ui.RefineryUtilities;

import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGEncodeParam;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

import java.awt.image.BufferedImage;
import java.awt.Font;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.FileImageOutputStream;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**
 * This is just a simple wrapper class to allow you to easily create and
 * display a chart with JFreeChart. The wrapper class itself only has the
 * standard method signatures for the class, as well as the functionality
 * for displaying a chart as an image, JPEG file, dialog, etc. Subclasses
 * should be created for specific chart types, overriding at least the
 * createChart(), addData(), and clearData() methods.
 * <br/><br/>
 * Tested with JFreeChart 1.0.14
 * <br/><br/>
 * 
 * version 1.5<br/>
 * January 2, 2009<br/>
 * Julian Robichaux, http://www.nsftools.com
 * 
 * @author Julian Robichaux, http://www.nsftools.com
 * @version 1.5
 */
public class GenericChartWrapper {
	protected Dataset data;
	protected JFreeChart chart;
	protected String chartTitle = "";
	protected String verticalAxisLabel = "";
	protected String horizontalAxisLabel = "";
	protected boolean dataChanged = false;
	protected Font font = null;
	protected boolean showLegend = false;
	protected double markerValue = 0.0;
	protected String markerLabel = "";
	
	
	public GenericChartWrapper () {
		clearData();
	}
	
	
	/**
	 * Give the chart a title.
	 */
	public void setTitle (String title) {
		chartTitle = title;
	}
	
	
	/**
	 * Set the axis labels, if any.
	 */
	public void setAxisLabels (String bottomLabel, String sideLabel) {
		horizontalAxisLabel = bottomLabel;
		verticalAxisLabel = sideLabel;
	}
	
	
	/**
	 * Add some new data to the chart. Subclasses should override
	 * this method with an appropriate Dataset type.
	 */
	public void addData (double value, String dataLabel, String groupName) {
		((DefaultCategoryDataset)data).addValue(value, groupName, dataLabel);
		dataChanged = true;
	}
	public void addData (double value, String dataLabel) {
		addData(value, dataLabel, "");
	}
	
	
	/**
	 * Add a marker line, for graphs that support markers.
	 */
	public void setMarker (double value, String label) {
		markerValue = value;
		markerLabel = label;
	}
	protected void drawMarker (CategoryPlot plot, Font font) {
		// this would also work for an XYPlot
		if (markerValue > 0) {
			Marker marker = new ValueMarker(markerValue);
			marker.setLabel(markerLabel);
			marker.setLabelFont(font);
			marker.setLabelAnchor(org.jfree.ui.RectangleAnchor.BOTTOM_RIGHT);
			marker.setLabelTextAnchor(org.jfree.ui.TextAnchor.TOP_RIGHT); 
			marker.setPaint(java.awt.Color.GRAY);
			//half.setStroke(new java.awt.BasicStroke(2.0f));
			marker.setStroke(new java.awt.BasicStroke(2.0f, 
					java.awt.BasicStroke.CAP_ROUND, 
					java.awt.BasicStroke.JOIN_ROUND,
					1.0f, new float[] {10.0f, 6.0f}, 0.0f
					));
			plot.addRangeMarker(marker);
		}
	}
	
	
	/**
	 * Clear any data you've already added. Subclasses should override
	 * this method with an appropriate Dataset type.
	 */
	public void clearData () {
		// dataset type will be different for different chart types
		data = new DefaultCategoryDataset();
		showLegend = true;
		dataChanged = true;
	}
	
	
	/**
	 * Indicate whether or not to show a legend for this chart.
	 */
	public void setShowLegend (boolean showLegend) {
		this.showLegend = showLegend;
	}
	
	
	/**
	 * You should always call dispose() when you're done with the chart.
	 * Subclasses should add additional cleanup code here.
	 */
	public void dispose () {
		try {
			data = null;
			chart = null;
		} catch (Exception e) {
		}
	}
	
	
	/**
	 * Create the chart itself. You need to override this for each subclass
	 * to create different chart types. This method should never be called
	 * directly; rather, it is meant to be called internally when any of the
	 * other methods need a chart to be drawn.
	 */
	protected void createChart () {
		if ((chart == null) || dataChanged) {
			// create the chart here. For various chart types, see:
			// http://www.jfree.org/jfreechart/api/javadoc/org/jfree/chart/ChartFactory.html
			// chart = ChartFactory.create...();
			dataChanged = false;
		}
	}
	
	
	/**
	 * Return the current chart as a BufferedImage.
	 */
	public BufferedImage getAsBufferedImage (int width, int height) {
		createChart();
		ChartRenderingInfo renderInfo = new ChartRenderingInfo(null);
		BufferedImage bi = chart.createBufferedImage (width, height, BufferedImage.TYPE_INT_RGB, renderInfo);
		renderInfo = null;
		return bi;
	}
	
	
	/**
	 * Save the current chart as a PNG file. For example:
	 *     mychart.sendToPngFile("c:\\test.png", 850, 500);
	 */
	public void sendToPngFile (String fileName, int width, int height) 
		throws FileNotFoundException, IOException {
		BufferedImage bi = getAsBufferedImage(width, height);
		ImageIO.write(bi, "png", new File(fileName));
		bi = null;
	}
	
	
	/**
	 * Save the current chart as a JPEG file. For example:
	 *     mychart.sendToJpgFile("c:\\test.jpg", 850, 500);
	 */
	public void sendToJpgFile (String fileName, int width, int height) 
		throws FileNotFoundException, IOException {
		BufferedImage bi = getAsBufferedImage(width, height);
		//ImageIO.write(bi, "jpg", new File(fileName));	// poor quality output
		
		Iterator iter = ImageIO.getImageWritersByFormatName("jpeg");
		ImageWriter writer = (ImageWriter)iter.next();
		ImageWriteParam iwp = writer.getDefaultWriteParam();
		iwp.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
		iwp.setCompressionQuality(1);
		
		File file = new File(fileName);
		FileImageOutputStream out = new FileImageOutputStream(file);
		writer.setOutput(out);
		IIOImage image = new IIOImage(bi, null, null);
		writer.write(null, image, iwp);
		
		out.close();
		writer.dispose();
		bi = null;
	}
	
	
	/**
	 * ONLY USE THIS FOR PRE-JDK 1.4.2 INSTALLS.
	 * It uses com.sun files for JPEG conversion (bad) instead of
	 * ImageIO (good).
	 * 
	 * Save the current chart as a JPEG file. For example:
	 *     mychart.sendToJpgFile("c:\\test.jpg", 850, 500);
	 */
	public void sendToJpgFileOLD (String fileName, int width, int height) 
		throws FileNotFoundException, IOException {
		BufferedImage bi = getAsBufferedImage(width, height);
		FileOutputStream out = new FileOutputStream(fileName);
		JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
		JPEGEncodeParam param = encoder.getDefaultJPEGEncodeParam (bi);
		
		param.setQuality(1.0f, false);
		encoder.encode(bi, param);
		out.close();
		bi = null;
	}
	
	
	/**
	 * Draw the current chart on an AWT Container (or any subclass thereof).
	 */
	public void drawOnPanel (java.awt.Container panel, int width, int height) {
		BufferedImage bi = getAsBufferedImage(width, height);
		panel.add(new JLabel( new ImageIcon( bi )));
		bi = null;
	}
	

	/**
	 * Display the current chart in a dialog window. A good value
	 * for width/height is 850 and 550 for a larger window, or 500 and 325
	 * for a smaller one.
	 */
	public void displayAsDialog (int width, int height) {
		// create a frame to hold everything
		JFrame frame = new JFrame(chartTitle);
		frame.setDefaultCloseOperation ( JFrame.DISPOSE_ON_CLOSE );
		
		// create the panel that displays the chart
		JPanel panel = new JPanel();
		drawOnPanel(panel, width, height);
		
		// add the panel to the frame
		frame.setContentPane(panel);
		frame.pack();
		
		// center the frame and display it
		RefineryUtilities.centerFrameOnScreen(frame);
		frame.setVisible(true);
		frame.toFront();
		
		panel = null;
		frame = null;
	}
	
	
}

