package com.nsftools.charthelper;

/*
 * copyright 2009 SNAPPS, Julian Robichaux, and Rob McDonagh
 * 
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 2.1 of the License, or
 *     (at your option) any later version.
 *                 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *     
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with this program.  If not, see http://www.gnu.org/licenses.
 */

import java.text.NumberFormat;
import java.text.ParsePosition;
import java.util.HashMap;
import java.util.LinkedHashMap;

/**
 * This is a Java bean used for passing chart data around to different
 * classes, and to store data for later reuse.
 * 
 * @author Julian
 *
 */
public class ChartDataBean {
	private String name = "";
	private String id = "";
	private boolean dataChanged = true;
	private int chartType = 0;
	private String label = "";
	private String markerLabel = "";
	private double markerValue = 0.0;
	private HashMap itemData = new LinkedHashMap();

	public static final int TYPE_PIECHART = 0;
	public static final int TYPE_BARCHART = 1;
	public static final int TYPE_LINECHART = 2;
	public static final int TYPE_BARCHART3D = 3;
	
	// used internally for string-to-number conversions
	private NumberFormat nf = NumberFormat.getNumberInstance();
	
	
	/**
	 * Reset all the data inside this bean object to default values.
	 */
	public void resetData () {
		name = "";
		id = "";
		dataChanged = true;
		chartType = 0;
		markerLabel = "";
		markerValue = 0.0;
		itemData = new LinkedHashMap();
	}
	
	/**
	 * Copy all the data from another ChartDataBean into this one.
	 * This is convenient if multiple things inside your code share
	 * a pointer to the same ChartDataBean and you want to use another
	 * ChartDataBean instead of this one but don't want to reset the
	 * pointer.
	 * 
	 * @param cdb
	 */
	public void copyData (ChartDataBean cdb) {
		if (cdb == null) {
			resetData();
		} else {
			name = cdb.getName();
			id = cdb.getId();
			dataChanged = true;
			chartType = cdb.getChartType();
			markerLabel = cdb.getMarkerLabel();
			markerValue = cdb.getMarkerValue();
			itemData = cdb.getItemData();
		}
	}
	
	// getters and setters. Hoorah.
	public String getLabel() {
		return label;
	}
	public void setLabel(String label) {
		this.label = label;
	}
	public int getChartType() {
		return chartType;
	}
	public void setChartType(int chartType) {
		this.chartType = chartType;
	}
	public boolean isDataChanged() {
		return dataChanged;
	}
	public void setDataChanged(boolean dataChanged) {
		this.dataChanged = dataChanged;
	}
	public String getName() {
		return name;
	}
	public void setName(String descriptiveName) {
		this.name = descriptiveName;
	}
	public String getId() {
		return id;
	}
	public void setId(String uniqueID) {
		this.id = uniqueID;
	}
	public String getMarkerLabel() {
		return markerLabel;
	}
	public void setMarkerLabel(String markerLabel) {
		this.markerLabel = markerLabel;
	}
	public double getMarkerValue() {
		return markerValue;
	}
	public void setMarkerValue(String markerValue) {
		setMarkerValue(getDoubleFromString(markerValue).doubleValue());
	}
	public void setMarkerValue(double markerValue) {
		this.markerValue = markerValue;
	}
	public void setMarker(String markerLabel, double markerValue) {
		setMarkerLabel(markerLabel);
		setMarkerValue(markerValue);
	}
	public HashMap getItemData() {
		return itemData;
	}
	public void setItemData(HashMap columnData) {
		this.itemData = columnData;
	}
	
	
	/**
	 * Remove all the chart data in this bean without resetting
	 * any of the other properties.
	 */
	public void clearItemData() {
		this.itemData = new LinkedHashMap();
	}
	
	
	// easy ways to add label/value or label/value/group values to
	// the internal HashMap of data
	public void addItemData(String label, String value) {
		addItemData(label, value, "");
	}
	public void addItemData(String label, double value) {
		addItemData(label, value, "");
	}
	public void addItemData(String label, String value, String group) {
		addItemData(label, getDoubleFromString(value), group);
	}
	public void addItemData(String label, double value, String group) {
		addItemData(label, new Double(value), group);
	}
	public void addItemData(String label, Double value, String group) {
		HashMap groupValues;
		
		if (itemData.containsKey(group)) {
			groupValues = (HashMap)itemData.get(group);
		} else {
			// LinkedHashMap maintains data order (HashMap does not)
			groupValues = new LinkedHashMap();
		}
		
		if (groupValues.containsKey(label)) {
			groupValues.put(label, value + (Double)groupValues.get(label));
		} else {
			groupValues.put(label, value);
		}
		
		itemData.put(group, groupValues);
	}

	
	private Double getDoubleFromString (String txt) {
		// for a good discussion on converting strings to numbers, see:
		//http://www.ibm.com/developerworks/java/library/j-numberformat/index.html
		try {
			String digits = "-.,0123456789";
			int i;
			
			// skip non-numeric leading characters (like dollar signs)
			// that will cause parsing problems
			for (i = 0; i < txt.length(); i++) {
				if (digits.indexOf(txt.charAt(i)) >= 0) {
					break;
				}
			}
			
			ParsePosition pp = new ParsePosition(i);
			Number n = nf.parse(txt, pp);
			return new Double(n.doubleValue());
		} catch (Exception e) {
			return new Double(0);
		}
	}
	
}
