package com.nsftools.charthelper;

/*
 * copyright 2009 SNAPPS, Julian Robichaux, and Rob McDonagh
 * 
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 2.1 of the License, or
 *     (at your option) any later version.
 *                 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *     
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with this program.  If not, see http://www.gnu.org/licenses.
 */

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.swing.JPanel;

import org.eclipse.swt.SWT;
import org.eclipse.swt.awt.SWT_AWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;


/**
 * This is the visible component that the chart gets drawn on.
 * We have a decision to make here: inherit from SWT Component
 * or create a brand new class. Here are some links that discuss
 * inheritance versus composition:
 * 
 * http://www.eclipse.org/articles/Article-Writing%20Your%20Own%20Widget/Writing%20Your%20Own%20Widget.htm
 * http://www.javaworld.com/javaworld/jw-11-1998/jw-11-techniques.html
 * 
 * In this case I chose to compose, if for no other reason than
 * to avoid thread access exceptions if I need to get or set the
 * ChartDataBean outside of a UIJob thread.
 * 
 * @author Julian Robichaux
 *
 */
public class ChartContainer {
	private Composite awtContainer = null;
	private java.awt.Frame awtFrame = null;
	private java.awt.Color awtColor = null;
	private JPanel panel = null;
	private java.awt.event.ComponentAdapter resizeListener = null;
	private ChartDataBean chartData = null;
	private GenericChartWrapper jchart = null;
	private int lastSize = 0;
	private String lastLabel = "";


	public ChartContainer () {
		awtColor = java.awt.Color.WHITE;
	}
	
	
	/**
	 * Draw the chart component on the specified Composite.
	 */
	public void draw (Composite parent) {
		this.draw(parent, null);
	}
	
	
	/**
	 * Draw the chart component on the specified Composite using a
	 * given GridLayout.
	 */
	public void draw (Composite parent, GridData gridData) {
		awtContainer = new Composite(parent, SWT.EMBEDDED | SWT.NO_BACKGROUND);
		awtContainer.setLayout(new FillLayout(SWT.VERTICAL));
		awtFrame = SWT_AWT.new_Frame(awtContainer);
		
		if (gridData != null) {
			awtContainer.setLayoutData(gridData);
		}
		
		// redraw when we resize
		resizeListener = new java.awt.event.ComponentAdapter() {
			public void componentResized(java.awt.event.ComponentEvent e) {
				redrawChart();
			}
		};
		awtFrame.addComponentListener(resizeListener);
		
	}
	
	
	/**
	 * Always try to remember to call dispose when you're disposing
	 * the parent Composite this was created in.
	 */
	public void dispose () {
		try {
			if (jchart != null)
				jchart.dispose();
			awtFrame.removeComponentListener(resizeListener);
			awtFrame.dispose();
			awtContainer.dispose();
		} catch (Exception e) {
			// whatever
		}
	}
	
	
	public void setBackgroundColor (Color swtColor) {
		awtColor = new java.awt.Color(swtColor.getRed(), 
				swtColor.getGreen(), swtColor.getBlue());
	}
	
	
	public ChartDataBean getChartData() {
		return chartData;
	}


	public void setChartData(ChartDataBean chartData) {
		this.chartData = chartData;
	}


	public Display getDisplay () {
		return awtContainer.getDisplay();
	}
	
	
	public int getWidth() {
		return awtFrame.getWidth();
	}
	
	public int getHeight() {
		int height = (int)(awtFrame.getHeight()*0.95);
		if (chartData.getChartType() == ChartDataBean.TYPE_PIECHART)
			height = (int)(awtFrame.getWidth()*0.7);    // make sure it stays round
		return height;
	}
	
	
	// HACK: the updated image won't show up unless we force 
	// the parent container to redraw by resizing.
	// Oddly, we have to call redrawChart twice (it gets
	// called when we issue the resize); otherwise we'll end up
	// seeing the previously drawn frame, not the current one.
	public void forceRedraw() {
		awtFrame.setSize(awtFrame.getWidth()-1, awtFrame.getHeight());
		awtFrame.setSize(awtFrame.getWidth()+1, awtFrame.getHeight());
	}
	
	
	/**
	 * Redraw code for the JPanel component, called whenever the
	 * sidebar app gets resized or the data changes.
	 * This MUST be called from within a UIJob thread, either directly
	 * or indirectly.
	 */
	public void redrawChart() {
		try {
			int width = getWidth();
			int height = getHeight();
			if ((width <= 0) || (height <= 0))
				return;
			if (chartData == null)
				return;
			
			String label = chartData.getLabel();
			HashMap itemData = chartData.getItemData();
			boolean redrawTwice = (java.lang.Math.abs(lastSize - (width+height)) > 50);
			
			// simple caching strategy, so we don't recreate the panel excessively
			if (	(panel != null) && 
					(java.lang.Math.abs(lastSize - (width+height)) < 2) &&
					(label.equals(lastLabel)) && (!chartData.isDataChanged()) ) {
				//System.out.println("reusing last panel");
				
			} else {
				//System.out.println("drawing " + label + ", width = " + width);
				
				if (jchart != null)
					jchart.dispose();
				
				int maxDataItemRatio = 45;
				if (chartData.getChartType() == ChartDataBean.TYPE_LINECHART) {
					jchart = new LineChartWrapper();
					maxDataItemRatio = 30;
				} else if (chartData.getChartType() == ChartDataBean.TYPE_BARCHART) {
					jchart = new BarChartWrapper();
				} else if (chartData.getChartType() == ChartDataBean.TYPE_BARCHART3D) {
					jchart = new BarChart3DWrapper();
				} else {
					jchart = new PieChartWrapper();
				}
				
				if (itemData.size() > 0) {
				    Iterator it = itemData.entrySet().iterator();
				    while (it.hasNext()) {
				    	Map.Entry entry = (Map.Entry) it.next();
				    	String groupName = (String)entry.getKey();
				    	HashMap dataMap = (HashMap)entry.getValue();
				    	
				    	Iterator it2 = dataMap.entrySet().iterator();
				    	while (it2.hasNext()) {
					    	Map.Entry data = (Map.Entry) it2.next();
					    	jchart.addData((Double)data.getValue(), (String)data.getKey(), groupName);
				    	}
				    	
				    	// arbitrarily hide the legend if there are lots of items
				    	if ((width / dataMap.size()) < maxDataItemRatio)
				    		jchart.setShowLegend(false);
				    }
				}
				
				panel = null;
				panel = new JPanel();
				panel.setBackground(awtColor);
				jchart.setTitle(label);
				jchart.setMarker(chartData.getMarkerValue(), chartData.getMarkerLabel());
				jchart.drawOnPanel(panel, width, height);
				
				lastSize = (width+height);
				lastLabel = label;
			    chartData.setDataChanged(false);
			}
			
			// remove any excess old panels, but make sure there's always
			// at least one old one on the stack (otherwise you sometimes
			// get a blank panel displayed if you switch windows and then
			// switch back).
			if (awtFrame.getComponentCount() > 1) {
				awtFrame.remove(0);
			}
			awtFrame.add(panel);
			
			// HACK, to deal with the fact that when the sidebar snaps open
			// from a collapsed state, it redraws very small (or not at all)
			if (redrawTwice) {
				//System.out.println("redrawing again");
				forceRedraw();
			}
			
		} catch(Exception e) {
			// whatever
			e.printStackTrace();
		}
		
	}
	
	
	/**
	 * Save the current chart as a PNG file. Note that this will use the
	 * last chart drawn by redrawChart(). If no chart is available, null 
	 * will be returned; if there is an error, it will be thrown;
	 * otherwise, this will return the name of the file that was written to.
	 */
	public String writeToFile (String fileName, int width, int height) 
	throws FileNotFoundException, IOException {
		if (jchart == null)
			return null;
		//jchart.sendToJpgFile(fileName, width, height);
		jchart.sendToPngFile(fileName, width, height);
		return fileName;
	}
	
	
}
