package com.ibm.lotuslabs.context.service.document;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.ISelectionService;
import org.eclipse.ui.IWindowListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartConstants;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

/**
 * Gives the ability to add a listener to know the current document / selection.
 * 
 * @author bleonard
 */
public class DocumentContextService implements ISelectionListener, IWindowListener, IPartListener {

	private static DocumentContextService instance;
	public static DocumentContextService getDefault() {
		// singleton pattern
		if(instance==null)
			instance = new DocumentContextService();
		return instance;
	}

	private List uriListeners = new ArrayList();
	private List selListeners = new ArrayList();
	private IWorkbenchWindow currentWindow;
	private IWorkbenchPart currentPart;
	private DocumentSelection currentSelection;
	
	public DocumentContextService() {
		super();
		PlatformUI.getWorkbench().addWindowListener(this);
		
		// start with current window
		IWorkbenchWindow win = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if(win!=null) {
			windowActivated(win);
		}
	}
	
	// our listener (URI changes)
	public void addListener(IDocumentContextListener listener) {
		uriListeners.add(listener);
	}
	// all selection changes
	public void addSelectionListener(IDocumentContextListener listener) {
		selListeners.add(listener);
	}
	public void removeListener(IDocumentContextListener listener) {
		uriListeners.remove(listener);
		selListeners.remove(listener);
	}
	
	
	
	private ISelectionService getSelectionService() {
		IWorkbenchWindow win = getWorkbenchWindow();
		if(win==null)
			return null;
		return win.getSelectionService();
	}
	private ISelection getSelection() {
		ISelectionService service = getSelectionService();
		if(service==null)
			return null;
		return service.getSelection();
	}
	private IWorkbenchWindow getWorkbenchWindow() {
		if(currentWindow!=null) 
			return currentWindow;
		// figure it out
		return PlatformUI.getWorkbench().getActiveWorkbenchWindow();
	}
	
	public DocumentSelection getDocumentSelection() {
		if(currentSelection!=null)
			return currentSelection;
		// figure it out
		IWorkbenchPart part = getWorkbenchPart();
		if(part!=null)
			return new DocumentSelection(part,getSelection());
		return null;
	}
	public IWorkbenchPart getWorkbenchPart() {
		if(currentPart!=null)
			return currentPart;
		// figure it out
		IWorkbenchWindow win = getWorkbenchWindow();
		if(win!=null) 
			return win.getPartService().getActivePart();
		return null;
	}

	
	
	// notified by eclipse selection listener
	public void selectionChanged(final IWorkbenchPart part, final ISelection selection) {
		currentPart = part;
		if(uriListeners.size()==0 && selListeners.size()==0)
			return;	// nothing in list		
		
		// only make this once
		final DocumentSelection con = new DocumentSelection(part, selection);
		// how different is this?
		boolean same = false;
		if(currentSelection!=null) {
			IDocumentContext[] old = currentSelection.getItems();
			IDocumentContext[] now = con.getItems();
			if(old.length==now.length) {
				same = true;
				for(int i=0; i<now.length; i++) {
					if(old[i].equals(now[i]))
						continue;
					// they are also the same if the URIs are the same
					URI oldURI = old[i].getURI();
					URI nowURI = now[i].getURI();
					if(oldURI==null || nowURI==null || !oldURI.equals(nowURI)) {
						same = false;
						break;
					}
				}
			}
		}
		// still always update the current selection
		currentSelection = con;
		
		if(same && selListeners.size()==0) {
			return; // all document listeners, just get out.
		}
		
		// who we are going to tell about it
		List notifyList = new ArrayList();
		notifyList.addAll(selListeners);	// these always want to know
		if(!same) {	// these only want to know if really different
			notifyList.addAll(uriListeners);
		}
		
		Iterator itr = notifyList.iterator();
		while(itr.hasNext()) {
			final Object o = itr.next();
			Job notify = new Job("Document Context Notification") {
				public IStatus run(IProgressMonitor monitor) { 
					if(o instanceof IDocumentContextListener) {
						((IDocumentContextListener)o).selectionChanged(part,con);
					}
					return Status.OK_STATUS;
				}
			};
			notify.setSystem(true);
			notify.schedule();
		}		
	}

	
	// IWindowListener
	public void windowActivated(IWorkbenchWindow w) {
		w.getSelectionService().addPostSelectionListener(this);
		w.getPartService().addPartListener(this);
		currentWindow = w;
		
		// start with current part
		IWorkbenchPart part = w.getPartService().getActivePart();
		if(part!=null) {
			partActivated(part);
		}
			
	}
	public void windowDeactivated(IWorkbenchWindow w) {
		w.getSelectionService().removePostSelectionListener(this);
		w.getPartService().removePartListener(this);
		currentWindow = null;		
	}
	public void windowOpened(IWorkbenchWindow arg0) {}	
	public void windowClosed(IWorkbenchWindow arg0) {}
	
	
	// listen for the title of the current part to change
	private IPropertyListener titleListener = new IPropertyListener() {
		public void propertyChanged(Object arg0, int arg1) {
			if(arg0 instanceof IWorkbenchPart && arg0==currentPart) {
				if(arg1==IWorkbenchPartConstants.PROP_TITLE) {
					IWorkbenchPart part = (IWorkbenchPart)arg0;
					ISelectionProvider provider = part.getSite().getSelectionProvider();
					ISelection sel = null;
					if(provider!=null)
						sel = provider.getSelection();
					selectionChanged(part, sel);
				}
			}
		}
	};

	
	// IPartListener
	private IWorkbenchPart activatingPart;
	public void partActivated(final IWorkbenchPart part) {
		currentSelection = null;
		activatingPart = part;
		Job job = new Job("Document Context Part Activation") {
			public IStatus run(IProgressMonitor monitor) {
				if(part==currentPart || part!=activatingPart || currentSelection!=null)
					return Status.CANCEL_STATUS;
				selectionChanged(part, null);
				part.addPropertyListener(titleListener);
				return Status.OK_STATUS;
			}
		};
		job.setSystem(true);
		// wait a little bit to see if it has it's own selection
		//job.schedule(1500);
	}
	public void partDeactivated(IWorkbenchPart part) {
		// in case we added it
		part.removePropertyListener(titleListener);
	}
	public void partBroughtToTop(IWorkbenchPart arg0) {}
	public void partClosed(IWorkbenchPart arg0) {}
	public void partOpened(IWorkbenchPart arg0) {}
	
	
	
}
