package acme.sidebarchart.views;

/*
 * copyright 2009 SNAPPS, Julian Robichaux, and Rob McDonagh
 * 
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 2.1 of the License, or
 *     (at your option) any later version.
 *                 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *     
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with this program.  If not, see http://www.gnu.org/licenses.
 */

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Vector;

import lotus.domino.Document;
import lotus.domino.NotesException;
import lotus.domino.Session;
import lotus.domino.View;
import lotus.domino.ViewEntry;
import lotus.domino.ViewEntryCollection;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.*;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.SWT;

import acme.sidebarchart.Activator;
import acme.sidebarchart.NotesHelper;
import acme.sidebarchart.NotificationHelper;
import acme.sidebarchart.ViewPartListener;
import com.nsftools.charthelper.*;

import com.ibm.lotuslabs.context.service.document.*;
import com.ibm.notes.java.api.util.NotesJob;
import com.ibm.notes.java.api.util.NotesPlatform;


public class ChartView extends ViewPart {
	private Color bgColor;
	private ChartContainer chart = null;
	private ViewDataBean viewData = null;
	private Label viewNameLabel;
	private Combo comboChartType;
	private Combo comboCatColumn;
	private Combo comboValueColumn;
	private Button radioAllDocs;
	private Button radioSelectedDocs;
	private Vector viewColumns = null;

	private DocumentContextService service;
	private IDocumentContextListener contextListener;
	private DocumentSelection lastSelection;
	private IWorkbenchPage workbenchPage;
	private ViewPartListener partListener;
	private NotificationHelper notifier;
	
	private HashMap viewDataCache = new HashMap();
	private String lastViewURI = "";
	private boolean viewChanged = true;
	
	
	/**
	 * The constructor.
	 */
	public ChartView() {
	}

	
	/**
	 * This is a callback to create the viewer and initialize it.
	 */
	public void createPartControl(Composite parent) {
		viewData = new ViewDataBean();
		viewData.setLabel("No Docs Selected");
		parent.setLayout(new GridLayout(2, false));
		
		// colors we can use later (backgrounds, etc.), for consistency
		bgColor = new Color(getViewSite().getShell().getDisplay(), 255, 255, 255);


		// draw all the labels and combo boxes that the user can select from
		Label label = new Label(parent, SWT.NULL);
		label.setText("View Name: ");
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		viewNameLabel = new Label(parent, SWT.WRAP);
		viewNameLabel.setText("No View Selected");
		viewNameLabel.setLayoutData(gridData);

		label = new Label(parent, SWT.NULL);
		label.setText("Chart Type: ");
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		comboChartType = new Combo(parent, SWT.READ_ONLY);
		comboChartType.add("Pie Chart");
		comboChartType.add("Bar Chart");
		comboChartType.add("Line Chart");
		comboChartType.add("Bar Chart 3D");
		comboChartType.setText(comboChartType.getItem(0));
		viewData.setChartType(ChartDataBean.TYPE_PIECHART);
		comboChartType.setLayoutData(gridData);
		comboChartType.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				viewData.setChartType(comboChartType.getSelectionIndex());
				viewData.setDataChanged(true);
				chart.forceRedraw();
			}
		});
		
		label = new Label(parent, SWT.NULL);
		label.setText("Category Column: ");
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		comboCatColumn = new Combo(parent, SWT.READ_ONLY);
		comboCatColumn.setLayoutData(gridData);
		comboCatColumn.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				viewData.setCategoryColumn(comboCatColumn.getSelectionIndex());
				processSelection(null, lastSelection);
			}
		});
		
		label = new Label(parent, SWT.NULL);
		label.setText("Value Column: ");
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		comboValueColumn = new Combo(parent, SWT.READ_ONLY);
		comboValueColumn.setLayoutData(gridData);
		comboValueColumn.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				viewData.setValueColumn(comboValueColumn.getSelectionIndex());
				processSelection(null, lastSelection);
			}
		});
		
		label = new Label(parent, SWT.NULL);
		label.setText("Selection Type: ");
		Composite buttonContainer = new Composite(parent, SWT.NONE);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		buttonContainer.setLayout(new RowLayout());
		buttonContainer.setLayoutData(gridData);
		
		radioAllDocs = new Button(buttonContainer, SWT.RADIO);
		radioAllDocs.setText("All Docs  ");
		radioAllDocs.setSelection(true);
		radioAllDocs.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				viewData.setSelectAll(true);		
				processSelection(null, lastSelection);
			}
		});
		radioSelectedDocs = new Button(buttonContainer, SWT.RADIO);
		radioSelectedDocs.setText("Selected Docs");
		radioSelectedDocs.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				viewData.setSelectAll(false);
				processSelection(null, lastSelection);
			}
		});
		
		
		// here's our chart component
		gridData = new GridData(GridData.FILL_BOTH);
		gridData.horizontalSpan = 2;
		chart = new ChartContainer();
		chart.setBackgroundColor(bgColor);
		chart.setChartData(viewData);
		chart.draw(parent, gridData);

		
		// document selection listener setup (from Domiclipse docviewer)
		contextListener = new IDocumentContextListener() {
			public void selectionChanged(IWorkbenchPart part, DocumentSelection selection) {
				processSelection(part, selection);
			}
		};
		
		service = DocumentContextService.getDefault();
		service.addSelectionListener( contextListener ) ;
		
		
		// workbench part visiblity listener (from Domiclipse docviewer)
		partListener = new ViewPartListener( this ) ;
		workbenchPage = Activator.getDefault().getWorkbench().getActiveWorkbenchWindow().getActivePage() ;
		workbenchPage.addPartListener( partListener ) ;

		notifier = new NotificationHelper(this);
		addMenuHelp();
	}


	/**
	 * Cleanup code when this app is shutdown.
	 */
	public void dispose() {
		// remove any listeners, kill any AWT/Swing objects
		try {
			service.removeListener(contextListener);
			workbenchPage.removePartListener(partListener);
			bgColor.dispose();
			chart.dispose();
		} catch (Exception ignored) {
			// whatever
		}
		super.dispose();
	}
	
	
	/**
	 * What happens when this app gets focus
	 */
	public void setFocus() {
		System.out.println("Setting focus");
		chart.redrawChart();
	}

	
	/**
	 * Adds an "About This App" menu item and action to the sidebar menu.
	 */
	private void addMenuHelp () {
		Action helpAction = new Action() {
			public void run() {
				String msg = notifier.readPluginFile("files/AboutSidebarChart.html");
				notifier.displayHTMLDialog("About This App", msg);
			}
		};
		helpAction.setText("About This App");
		helpAction.setToolTipText("About This Application");
		helpAction.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().
			getImageDescriptor(ISharedImages.IMG_OBJS_INFO_TSK));
		
		Action sendToFileAction = new Action() {
			public void run() {
				String[] filters = { "*.png" };
				String fileName = notifier.saveFileDialog("Save Chart To PNG File", filters);
				if (fileName != null) {
					try {
						chart.writeToFile(fileName, 500, 325);
						NotesHelper.LogToStatusLine("Chart successfully saved to " + fileName);
						if (notifier.displayYesNoDialog("Save Successful", 
								"The chart was successfully saved to " + fileName + 
								". Would you like to open the file?")) {
							Program.launch(fileName);
						}
					} catch (Exception e) {
						notifier.displayDialog("Error", 
								"There was an error saving the file: " + e.getLocalizedMessage());
					}
				}
			}
		};
		sendToFileAction.setText("Send To File...");
		sendToFileAction.setToolTipText("Send To File");
		
		IActionBars bars = getViewSite().getActionBars();
		IMenuManager manager = bars.getMenuManager();
		manager.add(helpAction);
		manager.add(sendToFileAction);
	}
	
	
	/**
	 * Get the selected docs in the active view. Called from the 
	 * IDocumentContextListener selectionChanged() event.
	 * I'm very very sorry this method is so long. I was sloppy.
	 * In a perfect world this should be broken up into discrete parts.
	 */
	public void processSelection (final IWorkbenchPart part, DocumentSelection selection) {
		if (part == ChartView.this) {
			return;		// we selected ourself!
		}
		
		if (selection == null) {
			return;		// should never happen?
		}
		
		lastSelection = selection;
		if ( !partListener.isVisible() ) {
			return;
		}
		
		// this is final so it can be accessed within a thread
		final IDocumentContext[] array = selection.getItems();
		
		// all of the Notes access needs to be done in a NotesJob thread
		NotesJob njob = new NotesJob("doc thread") {
			protected IStatus runInNotesThread(IProgressMonitor arg0) throws NotesException {
				int docCount = 0;
				String viewName = "No selection";
				ArrayList noteIDs = new ArrayList();
				View view = null;
				
				Session session = NotesPlatform.getInstance().getSession();
				for (int i = 0; i < array.length; i++) {
					//System.out.println("URI = " + array[i].getURI());
					// NOTE: array[0].getURI() is null for non-view selections
					Document doc = NotesHelper.getDocument( session, array[i].getURI() );
					if (doc != null) {
						try {
							docCount++;
							noteIDs.add(doc.getNoteID());
							
							if (view == null) {
								String viewString = NotesHelper.parseViewURI(array[i].getURI());
								view = NotesHelper.getView( session, viewString );
								if (view != null) {
									viewName = view.getName();
									if (!viewString.equals(lastViewURI)) {
										viewChanged = true;
										viewColumns = (Vector)view.getColumnNames().clone();
									}
									//view.recycle(); 	//don't recycle yet, we need the view later
								}
								lastViewURI = viewString;
							}
						} catch (NotesException ne) {
							ne.printStackTrace();
						} catch (Exception je) {
							je.printStackTrace();
						} finally {
							try {
								doc.recycle();
							} catch (Exception re) {
								re.printStackTrace();
							}
						}
					}
				}
				
				// if the view changed, store the current info in our cache
				String vURI = viewData.getId();
				if ((vURI.length() > 0) && (viewChanged)) {
					ViewDataBean oldViewData = new ViewDataBean();
					oldViewData.copyData(viewData);
					viewDataCache.put(vURI, oldViewData);
					viewData.resetData();
					
					// see if we already have this view cached. If so, get the
					// previous info
					if (viewDataCache.containsKey(lastViewURI)) {
						viewData.copyData( (ViewDataBean)viewDataCache.get(lastViewURI) );
					}
				}
				
				// get data for the ChartDataBean, to fill in the chart.
				HashMap lastColumnSet = new HashMap();
				lastColumnSet.putAll(viewData.getItemData());
				viewData.clearItemData();
				viewData.setLabel("No Docs Selected");
				viewData.setName(viewName);
				viewData.setId(lastViewURI);
				
				if (view != null) {
					ViewEntryCollection vec = view.getAllEntries();
					int catColNum = viewData.getCategoryColumn();
					int valColNum = viewData.getValueColumn();
					
					// if the category and value columns have been set,
					// get the data from those columns for the chart
					// (otherwise, just chart the selected docs versus
					// the total docs)
					if ((catColNum > 0) && (valColNum > 0)) {
						try {
							ViewEntry entry = vec.getFirstEntry();
							ViewEntry nextEntry = null;
							
							while (entry != null) {
								if (entry.isCategory() || entry.isConflict() || 
										entry.isTotal() || !entry.isValid()) {
									// skip if it's not a valid doc
								} else if (viewData.isSelectAll() || noteIDs.contains(entry.getNoteID())) {
									Vector v = entry.getColumnValues();
									viewData.addItemData(v.get(catColNum-1).toString(), 
											v.get(valColNum-1).toString());
								}
								nextEntry = vec.getNextEntry(entry);
								entry.recycle();
								entry = nextEntry;
							}
							
						} catch (NotesException ne) {
							ne.printStackTrace();
						} catch (Exception je) {
							je.printStackTrace();
						}
						
						if (viewData.isSelectAll())
							docCount = vec.getCount();
						if (!viewData.getItemData().equals(lastColumnSet))
							viewData.setDataChanged(true);
						
						if (viewData.getItemData().size() > 0) {
							HashMap catItems = (HashMap)viewData.getItemData().values().iterator().next();
							viewData.setLabel(docCount + " selected, " + catItems.size() + " categories");
						}

					} else {
						viewData.setLabel(docCount + " of " + vec.getCount() + " docs selected");
						viewData.addItemData("Selected", docCount);
						viewData.addItemData("Not Selected", (vec.getCount() - docCount));
					}
					
					if (vec != null)
						vec.recycle();
					view.recycle();
				}
				
				// we should redraw in a UI thread, like this
				System.out.println("array count = " + array.length + "; redrawing. docCount = " + docCount);
				final String vnText = viewName;
				Thread t = new Thread() { public void run() { 
					if ((viewColumns != null) && viewChanged) {
						// reset the choices for the column drop-down boxes
						comboCatColumn.removeAll();
						comboValueColumn.removeAll();
						viewChanged = false;
						
						comboCatColumn.add("");
						comboValueColumn.add("");
						for (int i = 0; i < viewColumns.size(); i++) {
							String colName = (String)viewColumns.elementAt(i);
							if (colName.trim().length() == 0) {
								colName = "Untitled Column";
							}
							colName = (i+1) + ": " + colName;
							comboCatColumn.add(colName);
							comboValueColumn.add(colName);
						}
						
						// if we've used this view before, this will reset the
						// combo selections to their previous state
						comboCatColumn.setText(comboCatColumn.getItem(viewData.getCategoryColumn()));
						comboValueColumn.setText(comboValueColumn.getItem(viewData.getValueColumn()));
						comboChartType.setText(comboChartType.getItem(viewData.getChartType()));
						if (viewData.isSelectAll()) {
							radioAllDocs.setSelection(true);
							radioSelectedDocs.setSelection(false);
						} else {
							radioAllDocs.setSelection(false);
							radioSelectedDocs.setSelection(true);
						}
					}
					
					// write out the view name (and call layout() 
					// on the parent to wrap the label, if necessary
					viewNameLabel.setText(vnText);
					viewNameLabel.getParent().layout(true, true);
					
					// and finally, redraw the chart with all the data
					// we've collected
					chart.forceRedraw();
					} };
				getViewSite().getShell().getDisplay().asyncExec(t);
				
				return Status.OK_STATUS;
			}
		};
		
		njob.schedule();
	}
	
}
