package acme.sidebarchart;

/*
 * copyright 2009 SNAPPS, Julian Robichaux, and Rob McDonagh
 * 
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 2.1 of the License, or
 *     (at your option) any later version.
 *                 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *     
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with this program.  If not, see http://www.gnu.org/licenses.
 */

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.net.URL;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.part.ViewPart;

/**
 * A few easy-to-use methods for displaying messages and prompts.
 * 
 * @author Julian
 *
 */
public class NotificationHelper {

	private Shell pluginShell = null;
	private String lastFileDir = null;
	
	
	public NotificationHelper (ViewPart part) {
		pluginShell = part.getViewSite().getShell();
	}
	
	
	/**
	 * Display text information to the user with a dialog box that
	 * has Yes and No buttons. Returns true if the user clicked Yes,
	 * false otherwise.
	 */
	public boolean displayYesNoDialog (String title, String msg) {
		MessageBox messageBox = new MessageBox(pluginShell, SWT.ICON_QUESTION
				| SWT.YES | SWT.NO);
		messageBox.setMessage(msg);
		messageBox.setText(title);
		int response = messageBox.open();
		if (response == SWT.YES)
			return true;
		else
			return false;
	}
	
	
	/**
	 * Display text information to the user with a dialog box.
	 * displayDialog() accepts text only, while displayHTMLDialog()
	 * can format HTML.
	 */
	public void displayDialog (String title, String msg) {
		MessageDialog.openInformation(pluginShell, title, msg);
	}
	
	
	/**
	 * Display HTML information to the user with a dialog box.
	 */
	public void displayHTMLDialog (String title, String html) {
		Shell shell = new Shell(pluginShell.getDisplay(), 
				SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
		shell.setText(title);
		
		shell.setLayout(new FillLayout());
	    Browser browser = new Browser(shell, SWT.NONE);
	    browser.setText(html);
	    
	    shell.open();
	}
	
	
	/**
	 * Read a text file that's bundled with this plugin and return it
	 * as a String. fileName is of the format "path/file name", like
	 * "META-INF/AboutThisApp.txt". Good for using with displayHTMLDialog()
	 * to show help info, etc.
	 */
	public String readPluginFile (String fileName) {
		try {
			URL fileUrl = Activator.getDefault().getBundle().getEntry(fileName);
			StringWriter sw = new StringWriter();
			BufferedReader in = new BufferedReader( 
					new InputStreamReader(fileUrl.openStream()));  
			
			String txt;
			while( (txt = in.readLine()) != null ) {  
			    sw.write(txt);  
			}  
			
			in.close();  
			sw.close();  
			
			return sw.toString(); 
		} catch (Exception e) {
			return "could not find file " + fileName;
		}
	}
	
	
	/**
	 * Display a "Save File" dialog and return the name of the selected file.
	 * This will return null if the dialog was cancelled. The "filters" param
	 * can be something like: 
	 * String[] filters = { "*.jpg;*.jpeg", "*.gif" };
	 */
	public String saveFileDialog (String title, String[] filters) {
		FileDialog fd = new FileDialog(pluginShell, SWT.SAVE);
        fd.setText(title);
        fd.setFilterPath(lastFileDir);
        fd.setFilterExtensions(filters);
        String fileName = fd.open();
        lastFileDir = fd.getFilterPath();
        return fileName;
	}
}
