package acme.sidebarchart;

/*
 * copyright 2009 SNAPPS, Julian Robichaux, and Rob McDonagh
 * 
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 2.1 of the License, or
 *     (at your option) any later version.
 *                 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *     
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with this program.  If not, see http://www.gnu.org/licenses.
 */

import java.net.URI;
import java.util.StringTokenizer;

import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import lotus.domino.Database;
import lotus.domino.Document;
import lotus.domino.NotesException;
import lotus.domino.Session;
import lotus.domino.View;

/**
 * This class contains a few static methods that are useful for
 * interacting with the Lotus Notes client. The methods should
 * always be called from within a NotesJob thread.
 * 
 * @author Julian
 *
 */
public class NotesHelper {
	
	/**
	 * Given an active Notes Session and a document URI, return the
	 * document that the URI points to, if any. This will return null
	 * if the URI does not point to a document, or if it points to a
	 * document that is in a local database that is not in your Notes
	 * data directory.
	 * 
	 * @param session
	 * @param uri
	 * @return
	 */
	public static Document getDocument(Session session, URI uri) {
		// NOTE: session.resolve will fail on local databases that are not
		// in the Notes data directory.
		Document doc = null;
		if (uri != null) {
			try {
				Object o = session.resolve( uri.toString() );
				if(o!=null) {
					if(o instanceof View) {
						((View)o).recycle(); 
					} else if(o instanceof Database) {
						((Database)o).recycle(); 
					} else if(o instanceof Document) {
						doc = (Document) o; 
						if (doc.hasItem("$Collation")) {
							// for some reason categories get retrieved as a doc
							// representing the View design note. This would be
							// an interesting hack if it's what we wanted, but
							// right now it's not.
							doc.recycle();
							doc = null;
						}
					}
				}
			} catch (NotesException ne) {
				ne.printStackTrace(); 
			} catch (Exception e) {
				e.printStackTrace(); 
			} 
		} 

		return doc;
	}

	
	/**
	 * A very crude way to convert a Notes doc URI to a URI for that doc's view.
	 *  
	 * @param docUri
	 * @return
	 */
	public static String parseViewURI (URI docUri) {
		String viewString = "";
		if (docUri != null) {
			viewString = docUri.toString();
			int pos = 0;
			int count = -1;
			while (pos >= 0) {
				count++;
				pos = viewString.indexOf('/', pos+1);
			}
			
			if (count > 4) {
				pos = viewString.lastIndexOf("/");
			} else {
				pos = viewString.indexOf("?");
			}
			
			if (pos > 0)
				viewString = viewString.substring(0, pos);
			
			viewString += "?OpenView";
		}
		return viewString;
	}
	
	
	/**
	 * Given an active Notes Session and a View URI, return the
	 * View that the URI points to, if any. This will return null
	 * if the URI does not point to a valid view.
	 * 
	 * @param session
	 * @param viewURIString
	 * @return
	 */
	public static View getView (Session session, String viewURIString) {
		View view = null;
		if ((viewURIString != null) && (viewURIString.length() > 0)) {
			try {
				Object o = session.resolve( viewURIString );
				if(o != null) {
					if(o instanceof Document) {
						((Document)o).recycle(); 
					} else if(o instanceof View) {
						view = (View)o;
						view.setAutoUpdate(false);
					} else if(o instanceof Database) {
						((Database)o).recycle(); 
					} else {
						//System.out.println("view resolve got an unknown object");
					}
				}
			} catch (NotesException ne) {
				ne.printStackTrace(); 
			} catch (Exception e) {
				e.printStackTrace(); 
			} 
		} 

		return view;
	}

	
	/**
	 * A way to write messages to the Notes status bar, from:
 	 * http://blog.balfes.net/index.php?entry=entry080305-120658
	 * 
	 * @param msg
	 */
	public static void LogToStatusLine (final String msg) {
		Display display = Display.getDefault();
		display.asyncExec(new Runnable(){
			public void run() {
				IWorkbenchWindow win =  PlatformUI.getWorkbench().getActiveWorkbenchWindow();
				IWorkbenchPage page = win.getActivePage();
				IWorkbenchPart part = page.getActivePart();
				IWorkbenchPartSite site = part.getSite();

				if (site instanceof IViewSite){
					IViewSite vSite = ( IViewSite ) site;

					IActionBars actionBars =  vSite.getActionBars();
					if( actionBars == null )
						return ;

					IStatusLineManager statusLineManager =
						 actionBars.getStatusLineManager();
					if( statusLineManager == null )
						return ;

					statusLineManager.setMessage( msg);
				}				
			}
				
		});
	}
	
	
}
